//
//  Visual_modelV2e.cpp
//  SimpleCrowdingModel
//
//  Created by David Kieras on 5/3/19.
//  Copyright © 2019 University of Michigan. All rights reserved.
//

/*
Version V2e. This version includes internal options for the order in which objects are processed
for crowding computation. This corrects an odd situation in which objects were processed in the
last order that the Strategy had left the Search_objects container, which doesn't make much sense,
becaue the eye was in a different location.

Version V2d. This version eliminates the Search_object current_crpwding member variable which
was being incorrectly set during computing the crowding group to show that an flanking object was crowded
when it not necessarily was crowded. The size of an object's crowding group > 0 means that it has crowders.
This means the current_crowded member variable is redundant.

Version V2c. This version adds options to enable/disable the features added since the original V2.
The option controls are in the Program_constants module.

Version V2b. This version adds an information-gained score for each object
that gives the expected number of objects whose properties would be revealed if this object is fixated;
the expected number is determined by the availability of the surrounding objects assuming that the
point of fixation is a directly on this object.

Version V2a. This version modifies treatment of objects that are in perceptual store state:
The property can be overwritten by a non-blank property, but not by a blank property.
This means that the scrambling can replace a property in perceptual store state,
but only by an available property - a previously available property is still "sticky"
if object falls into peripheral vision, but become subject to crowding anew. 190219

Version V2. This version attempts to fix the problem that V1 disrupted the memory of
the properties for previously fixated objects.

Version that reorders the objects so that crowding computations are performed in a well-defined order -
previous version performed the computation in whatever order the strategy last left the objects in, which
is semi-haphazard.

NOTE: This module is supposed to be independent of the Strategy model -- the Strategy model can ask for the eye to be moved
to an object's location, but this module computations are based only on the current location of the eye, not on the object
whose location was used to fixate.  So if this module reorders the objects in its container (a vector, for speed), the Strategy
cannot assume that an interator or reference to an object in the vector is still valid - it must re-find the object.
*/

#include "Program_constants.h"
#include "Visual_modelV2e.h"
#include "Search_object.h"
//#include "EPICLib/Random_utilities.h"
#include "Random_utilities.h"
#include <iostream>
#include <sstream>
#include <cassert>
#include <cmath>


using namespace std;

// This should be moved to EPICLib as being generally useful
double normal_cdf(double x, double mean, double sd);

bool LinearConstSD_available(double eccentricity, GU::Size size, double a, double b, double sd);
double LinearConstSD_detection_probability(double eccentricity, GU::Size size, double a, double b, double sd);

enum class Crowding_comp_order_e {Last_order, From_furthest, From_closest, Random};

Crowding_comp_order_e crowding_comp_order = Crowding_comp_order_e::From_closest;

/* This model implements crowding for each object in a way that maximizes crowding
and statistically independently of the number of crowding objects.
*/
string Visual_model::get_id_string()
{
    ostringstream oss;
//   oss << "Visual model V2d";
//    oss << "Visual model V2dx";
//    oss << "Visual model V2dy";
//    oss << "Visual model V2dz";
    oss << "Visual model V2e";
	if(!apply_crowding_effect)
    	oss << " with no crowding";
	else
		switch (crowding_comp_order) {
			case Crowding_comp_order_e::Last_order: {
				oss << " with crowding probability cp if (n_crowders > 0), crowding applied to each object in last order, asymetrically between objects differing in eccentricity.\n"; // V2d original
				break;
				}
			case Crowding_comp_order_e::From_furthest: {
				oss << " with crowding probability cp if (n_crowders > 0), crowding applied to each object from greatest eccentricity, asymetrically between objects differing in eccentricity.\n"; // V2dx
				break;
				}
			case Crowding_comp_order_e::From_closest: {
				oss << " with crowding probability cp if (n_crowders > 0), crowding applied to each object from least eccentricity, asymetrically between objects differing in eccentricity.\n"; // V2dy
				break;
				}
			case Crowding_comp_order_e::Random: {
				oss << " with crowding probability cp if (n_crowders > 0), crowding applied to each object in random order, asymetrically between objects differing in eccentricity.\n";// V2dz
				break;
				}
			default:
				assert(!"undefined Crowding_comp_order");
			};

		oss << "Once object properties available with no crowders, ";
        switch (perceptual_store_scramble_policy) {
            case Perceptual_store_scramble_policy_e::NO_OVERWRITES: {
                 oss << "its properties are no longer scrambled.\n";
               break;}
            case Perceptual_store_scramble_policy_e::NON_BLANK_OVERWRITES: {
                oss << "scrambling can replace with only non-blank property.\n";
                break;}
            case Perceptual_store_scramble_policy_e::ALL_OVERWRITES: {
                oss << "scrambling can still change its properties.\n";
                break;}
            }

    if(unitary_shape_c)
        oss << "Unitary shape.";
    else
        oss << "Shape with " << n_shape_parts_c << " parts.";
    
    if(saccade_noise_present_c)
        oss << " Saccade noise present.";
    else
        oss << " No saccade noise.";

    return oss.str();
}
// initialize the eye position and process the first set of objects
void Visual_model::initialize()
{
    n_fixations = 0;
    eye_location = GU::Point(0., 0.);
    // special case set of eccentricity
    for(auto& obj : search_objects)
    	obj.eccentricity = cartesian_distance(eye_location, obj.location);
	process_search_objects(); // present display before update
	if(processed_visual_trace_output || detailed_visual_trace_output) {
		cout << "*** after update for initial display; eye at: " << eye_location << endl;
		print_objects();
		}
}

// move eye to the new location using noisy saccades, then process the objects
double Visual_model::move_eye(Search_object& obj)
{
//    if(strategy_trace_output)
//        cout << "Visual: before saccade to " << obj.ID << " at " << obj.location << endl;
    double time = make_saccade(obj.location); // make the saccade
    n_fixations++;
    process_search_objects();
//    if(strategy_trace_output)
//        cout << "Visual: after saccade to " << obj.ID << " eccentricity is " << obj.eccentricity << endl;
	if(processed_visual_trace_output || detailed_visual_trace_output) {
		cout << "Visual: after update for eye fixation " << n_fixations << " on object " << obj.ID << "; eye at: " << eye_location << endl;
    	print_objects();
    	}
    return time;
}

double Visual_model::make_saccade(GU::Point new_location)
{
    double gain = .95;
    double s = 0.10;
    double saccade_angle_sd = 1;

    if(!saccade_noise_present_c) {
        gain = 1.0;
        s = 0.0;
        saccade_angle_sd = 0;
        }

    double time = 0;
    
    GU::Polar_vector move_vector(eye_location, new_location);
    // no move if eye is actually already there, essentially
    if(move_vector.r < 0.1){
        move_vector.r = 0.0;
        time = em_time_intercept_c; // ???
        return time;
        }
    assert(move_vector.r > 0.);
    // modify r using the saccade distance parameters - gain and s, sample actual distance
    double base_distance = move_vector.r * gain;
    double sd = base_distance * s;
    double actual_distance = base_distance;
    // if shortfall_distance is 50, .1 times that gives sd of 5, 3 sds out is 15 degrees, saccade of 75 degrees
    // require that actual distance is > 0.
    if(sd > 0.0) {
        do {actual_distance = trimmed_normal_random_variable(base_distance, sd, 3.0);
            } while (actual_distance <= 0.0);
        }

    move_vector.r = actual_distance;

    // modify theta using noise parameter independent of r
    double saccade_angle_SD = GU::to_radians(saccade_angle_sd);
    double actual_angle = move_vector.theta;
    if(saccade_angle_SD > 0.0) {
        actual_angle = trimmed_normal_random_variable(actual_angle, saccade_angle_SD, 3.0);
        }
    move_vector.theta = actual_angle;

    // translate the eye location using the computed move_vector
    eye_location = eye_location + move_vector;
    time = em_time_intercept_c + em_time_slope_c * move_vector.r;
    return time;
}


void Visual_model::process_search_objects()
{
    if(detailed_visual_trace_output) {
        cout << "\nVisual: before update of sensory properties; eye at: " << eye_location << endl;
        print_objects();
        }
    update_sensory_properties();
    if(detailed_visual_trace_output) {
		cout << "Visual: after update of sensory properties; eye at: " << eye_location << endl;
		print_objects();
		}
    update_perceptual_properties();
    if(detailed_visual_trace_output) {
        cout << "Visual: after update of perceptual properties; eye at: " << eye_location << endl;
        print_objects();
        }
    if(apply_crowding_effect) {
		switch (crowding_comp_order) {
			case Crowding_comp_order_e::Last_order: {
				// V2d original
				break;
				}
			case Crowding_comp_order_e::From_furthest: {
				// V2dx sort objects from largest eccentricity to smallest before crowding
				sort(search_objects.begin(), search_objects.end(),
					[this](const auto& obj1, const auto& obj2)
					{return (GU::cartesian_distance(obj1.location, eye_location) > 	GU::cartesian_distance(obj2.location, eye_location));});
				break;
			case Crowding_comp_order_e::From_closest: {
				// V2dy sort objects from smallest eccentricity to largest before crowding
				sort(search_objects.begin(), search_objects.end(),
					[this](const auto& obj1, const auto& obj2)
					{return(GU::cartesian_distance(obj1.location, eye_location) < 	GU::cartesian_distance(obj2.location, eye_location));});
				break;
				}
				}
			case Crowding_comp_order_e::Random: {
				// V2dz put objects in a random order before crowding
				shuffle(search_objects.begin(), search_objects.end(), get_Random_engine());
				break;
				}
			default:
				assert(!"undefined Crowding_comp_order");
			};

        if(detailed_visual_trace_output || crowding_visual_trace_output) {
            cout << "Visual: after reordering of search objects relative to " << eye_location << endl;
            print_objects();
            }
        find_crowders();
        scramble_perceptual_properties();
        if(detailed_visual_trace_output || crowding_visual_trace_output) {
            cout << "Visual: after scramble of perceptual properties; eye at: " << eye_location << endl;
            print_objects();
            }
        // now if all of a currently uncrowded object's properties are known, mark it as known
        update_crowding_status();
        }
    
    // always check tha target status for instrumentation purposes
    check_target_status(condition, search_objects);
    
    if(compute_Shape_information_gain_g) {
        update_Shape_information_gain();
        if(detailed_visual_trace_output) {
            cout << "Visual: after update_Shape_information_gain; eye at: " << eye_location << endl;
            print_objects();
            }
        }
    
    if(compute_Feature_information_gain_g) {
        update_Feature_information_gain();
        if(detailed_visual_trace_output) {
            cout << "Visual: after update_Feature_information_gain; eye at: " << eye_location << endl;
            print_objects();
            }
        }

//    if(processed_visual_trace_output || detailed_visual_trace_output || target_status_trace_output) {
//        check_target_status(condition, search_objects);
//        }
}

void Visual_model::print_objects() const
{
	for(auto& obj : search_objects)
		cout << obj << endl;
}


// set the sensory properties - depend only on property, eccentricity, and size - not crowding
void Visual_model::update_sensory_properties()
{
//    cout << "spat_config2: "; segs_out(spat_config2); cout << "\n";
//    cout << "spat_config5: "; segs_out(spat_config5); cout << "\n";

    for(auto& obj : search_objects) {
        obj.eccentricity = cartesian_distance(obj.location, eye_location);
        obj.sens_color = LinearConstSD_available(obj.eccentricity, obj.size, common_intercept_c, color_slope_g, common_sd_c) ? obj.phys_color : '.';
        obj.sens_orientation = LinearConstSD_available(obj.eccentricity, obj.size, common_intercept_c, orientation_slope_g, common_sd_c) ? obj.phys_orientation : '.';
        if(unitary_shape_c) {
            obj.sens_shape = LinearConstSD_available(obj.eccentricity, obj.size, common_intercept_c, shape_slope_g, common_sd_c) ? obj.phys_shape : '.';
            }
        else {
            // for each segment, sample availability, leave as blank if not present in phys
            // synthesize phys_spat_config here instead of having it constructed in the Search object
            Shape_parts_t phys_spat_config;
            if(obj.phys_shape == '2')
                phys_spat_config = Search_object::spat_config2;
            else if(obj.phys_shape == '5')
                phys_spat_config = Search_object::spat_config5;
            else
                phys_spat_config = Search_object::spat_configNull; // no 2 or 5 shape specified, have null spat_config pattern

            // now sample availability for each segment in the physical spat config
            for(int i = 0; i < obj.sens_spat_config.size(); i++) {
                // using shape_slope_g for each shape_part
                obj.sens_spat_config[i] = (LinearConstSD_available(obj.eccentricity, shape_part_size_c, common_intercept_c, shape_slope_g, common_sd_c)) ?
                    phys_spat_config[i] :
                    '.';
                }
            }
        }
}



// simply move known sensory property to perceptual property - if sensory property is unknown, leave perceptual property alone
void Visual_model::update_perceptual_properties()
{
    // update the object properties for each object
    for(auto& obj : search_objects) {
        // simple idea: we always update perceptual properties according to now available sensory properties
        // with the perceptual properties that are present being sticky.
        // cases:
        // sens perc -> perc
        //  .   .       . (unchanged)
        //  .   x       x (unchanged) sticky property
        //  x   .       x (update now available property)
        //  x   x       x (unchanged)
        //  x   y       x (correct bogus previous property)
        // non-null sens_color unconditionally becomes current perceptual color
        if(obj.sens_color != '.') {
            obj.perc_color = obj.sens_color;
            }
        if(obj.sens_orientation != '.') {
            obj.perc_orientation = obj.sens_orientation;
           }
        if(unitary_shape_c) {
            if(obj.sens_shape != '.') {
                obj.perc_shape = obj.sens_shape;
                }
            }
        else {
            for(int i = 0; i < obj.sens_spat_config.size(); i++) {
                if(obj.sens_spat_config[i] != '.')
                    obj.perc_spat_config[i] = obj.sens_spat_config[i];
                }
        
            // set the perc_shape unitary property based on the perc_spat_config pattern
            // object, from-field, to-field
            recognize_shape(obj, &Search_object::perc_shape, &Search_object::perc_spat_config);
            }
        // other cases are automatic
    }
}



// assuming update_sensory_properties already called to update eccentricity
// go through all search objects and find out which other objects are crowding them : the crowding group
// note that the current object is a member of the crowding group, so size of group is >= 1
// TODO: Fix that - no reason that uncrowded object represented that way - see later code checking for > 1 for crowding group size
// if no crowders, mark the object is_uncrowded unless - sticky property
/*
Normally, if an object is crowded by another object, or crowds another object, we scramble properties with the other object.
If it neither is crowded, nor crowds, any other objects, is become uncrowded and thereafter, its assigned properties will be sticky
and not changed even if it crowds in the future, unless the properties are different while it is currently uncrowded.
???  TODO UPDATE THIS COMMENT
*/
void Visual_model::find_crowders()
{
    // first, set the current_crowding to false, we'll update this while finding all the crowding groups
//    for (auto& obj : search_objects)
//       obj.current_crowding = false;

    // now, for each object, find out if it has any crowders, and mark both this and the crowding objects as being currently crowded
    for(int i = 0; i < search_objects.size(); i++) {
        Search_object& obj1 = search_objects[i];
        obj1.crowder_group.clear();
       double critical_spacing = bouma_fraction_c * obj1.eccentricity; // bouma function
        for(int j = 0; j < search_objects.size(); j++) {
            if(i == j)
                continue;
            Search_object& obj2 = search_objects[j];
            double obj_spacing = cartesian_distance(obj1.location, obj2.location);
            if(obj_spacing <= critical_spacing) {
                // add obj2's index to obj1's crowding group
                obj1.crowder_group.push_back(j);
                // mark that obj1 is currently crowded by another object
//                obj1.current_crowding = true;
                // mark that obj2 is currently crowded by another object
//                obj2.current_crowding = true;
                }
            }
        // the crowder group is empty if this object has no crowders, > 1 if it does
        // put this object's index into the crowding group
        if(obj1.crowder_group.size() > 0) {
            obj1.crowder_group.push_back(i); // put index of this object into the list
            }
       }
}


// assumes that update_perceptual_properties and find_crowders has already been called
// scramble if there are crowders
void Visual_model::scramble_perceptual_properties()
{
    // scramble the object properties for each object
    for(auto& obj : search_objects) {
    
        if(detailed_visual_trace_output) {
            cout << "Visual: " << obj.ID << " has " << obj.crowder_group.size() << " objects in its crowder group {";
            for(int crowder_index : obj.crowder_group) {
                cout << ' ' << search_objects[crowder_index].ID;
                }
            cout << " }" << endl;
            }
        // either no crowders for the object, or this object and at least one other object
        assert(obj.crowder_group.size() == 0 || obj.crowder_group.size() >= 2);
        if(obj.crowder_group.size() > 1) {
            switch(condition) {
                case Condition_e::CSF:
                    if(biased_coin_flip(color_crowding_probability_g))
                        scramble_perc_properties_by_permutation(obj, &Search_object::perc_color);
                    break;
                case Condition_e::COC:
                    if(biased_coin_flip(color_crowding_probability_g))
                        scramble_perc_properties_by_permutation(obj, &Search_object::perc_color);
                    if(biased_coin_flip(orientation_crowding_probability_g))
                        scramble_perc_properties_by_permutation(obj, &Search_object::perc_orientation);
                    break;
                case Condition_e::SHP:
                    if(unitary_shape_c) {
                        if(biased_coin_flip(shape_crowding_probability_g))
                            scramble_perc_properties_by_permutation(obj, &Search_object::perc_shape);
                        }
                    else {
                        if(biased_coin_flip(shape_crowding_probability_g))
                            scramble_perc_spat_configs_by_permutation(obj);
                        }
                    break;
                default:
                    assert(!"unknown condition in update_perceptual_properties");
                    break;
        }

            } // end of crowder_group.size() > 1 case
        } // end of each-object loop
}

// scramble the property pointed to by perc_prop_ptr
// but if either obj, or an object in the crowding group is marked as known_perceptual_properties, do not modify its properties
// but its properties are still scrambled with those of other objects and so might be assigned to some other object
void Visual_model::scramble_perc_properties_by_permutation(Search_object& obj, char Search_object::*perc_prop_ptr)
{
    // crowder_group has indicies for crowding objects
    // current object is included in the list
    // get the properties of objects in the crowding grouping and scramble their order
    vector<char> scrambled_properties;
    for(int i = 0; i < obj.crowder_group.size(); i++) {
        int obj_idx = obj.crowder_group[i];
        if(detailed_visual_trace_output) cout << search_objects[obj_idx].ID << ' ' << 	search_objects[obj_idx].*perc_prop_ptr << endl;
        scrambled_properties.push_back(search_objects[obj_idx].*perc_prop_ptr);
        }
    if(detailed_visual_trace_output) {
        for(auto c : scrambled_properties)
            cout << c << ' ';
        cout << endl;
        }
    shuffle(scrambled_properties.begin(), scrambled_properties.end(), get_Random_engine());
    assert(obj.crowder_group.size() == scrambled_properties.size());
    if(detailed_visual_trace_output) {
        for(auto c : scrambled_properties)
            cout << c << ' ';
        cout << endl;
        }

    // go through the crowder list and assign the property in the scrambled_properties list
    // but do not assign to a blank property to an object with known_perceptual_properties 190219
    for(int i = 0; i < obj.crowder_group.size(); i++) {
        int obj_idx = obj.crowder_group[i];
        Search_object& obj = search_objects[obj_idx];
        // if object is not in perceptual store state, always scramble the properties
        if(!obj.known_perceptual_properties) {
            obj.*perc_prop_ptr = scrambled_properties[i];
            if(detailed_visual_trace_output || (obj.is_target && target_status_trace_output))
                cout << obj.ID << "<-" << obj.*perc_prop_ptr << endl;
                }
        else {  // if the object is in perceptual store state, scramble depending on the policy
            switch (perceptual_store_scramble_policy) {
                case Perceptual_store_scramble_policy_e::NO_OVERWRITES: {
                    // do nothing
                    break;}
               
            case Perceptual_store_scramble_policy_e::NON_BLANK_OVERWRITES: {
                // scramble only if the property is non-blank
                // check: assign if either not known_perceptual_properties OR non-blank property
                if(scrambled_properties[i] != '.') {
                        obj.*perc_prop_ptr = scrambled_properties[i];
                    if(detailed_visual_trace_output || (obj.is_target && target_status_trace_output))
                            cout << obj.ID << "<-" << obj.*perc_prop_ptr << endl;
                        }
                    break;}
                
            case Perceptual_store_scramble_policy_e::ALL_OVERWRITES: {
                // scramble the property regardless
                obj.*perc_prop_ptr = scrambled_properties[i];
                if(detailed_visual_trace_output || (obj.is_target && target_status_trace_output))
                    cout << obj.ID << "<-" << obj.*perc_prop_ptr << endl;
                break;}
					
			default: {
				assert(!"Unknown Perceptual_store_scramble_policy in Visual_model");
				break;}
                } // end of switch
            } // end of else
        } // end of for
}


// scramble the property pointed to by perc_prop_ptr
void Visual_model::scramble_perc_spat_configs_by_permutation(Search_object& obj)
{
    // crowder_group has indicies for crowding objects
    // current object is included in the list
    // get the properties of objects in the crowding grouping and scramble their order
    if(detailed_visual_trace_output) {
        cout << obj.ID << " has " << obj.crowder_group.size() << " objects in its crowder group {";
        for(int crowder_index : obj.crowder_group) {
            cout << ' ' << search_objects[crowder_index].ID;
            }
        cout << " }" << endl;
        }

    // the segment properties are all lumped together because it is assumed that each property has a location as a subpart of the object.
    vector<char> scrambled_properties;
    for(int i = 0; i < obj.crowder_group.size(); i++) {
        int obj_idx = obj.crowder_group[i];
        for(int j = 0; j < n_shape_parts_c; j++) {
             scrambled_properties.push_back(search_objects[obj_idx].perc_spat_config[j]);
             }
//        if(detailed_visual_trace_output) cout << search_objects[obj_idx].ID << ' ' << search_objects[obj_idx].*perc_prop_ptr << endl;
        }
    if(detailed_visual_trace_output) {
        for(auto c : scrambled_properties)
            cout << c << ' ';
        cout << endl;
        }
    shuffle(scrambled_properties.begin(), scrambled_properties.end(), get_Random_engine());
    assert(scrambled_properties.size() == obj.crowder_group.size() * n_shape_parts_c);
    if(detailed_visual_trace_output) {
        for(auto c : scrambled_properties)
            cout << c << ' ';
        cout << endl;
        }

    // go through the crowder list and assign the property in the scrambled_properties list
    // but do not assign blank property to an object with known_perceptual_properties
    // !!! NOTE THIS HAS NOT BEEN UPDATED AS ABOVE 190219
    for(int i = 0; i < obj.crowder_group.size(); i++) {
        int obj_idx = obj.crowder_group[i];
        Search_object& obj = search_objects[obj_idx];
        if(!obj.known_perceptual_properties) {
            if(detailed_visual_trace_output || (obj.is_target && target_status_trace_output)) {
                cout << obj.ID << ' '; obj.print_perc_spat_config(cout);
                cout << " -> ";
                }
            for(int j = 0; j < n_shape_parts_c; j++) {
                int prop_idx = i*n_shape_parts_c + j;
                assert(prop_idx < scrambled_properties.size());
                obj.perc_spat_config[j]= scrambled_properties[prop_idx];
                }
            if(detailed_visual_trace_output || (obj.is_target && target_status_trace_output)) {
                cout << obj.ID << ' '; obj.print_perc_spat_config(cout);
                cout << "\n";
                }
            }
        }
    // update each object's shape property based on segments - do here to simplify matching rules
    // note performed on all objects
    for(int i = 0; i < obj.crowder_group.size(); i++) {
        int obj_idx = obj.crowder_group[i];
        recognize_shape(search_objects[obj_idx], &Search_object::perc_shape, &Search_object::perc_spat_config);
        }
}

// set the shape unitary property based on the spat_config pattern
void Visual_model::recognize_shape(Search_object& obj, char Search_object::*to_ptr, Shape_parts_t Search_object::*from_ptr)
{
     obj.*to_ptr = Search_object::parts_to_shape_map[obj.*from_ptr];
/*
        if(obj.*from_ptr == Search_object::spat_config2)
            obj.*to_ptr = '2';
        else if(obj.*from_ptr == Search_object::spat_config5)
            obj.*to_ptr = '5';
        else if(obj.*from_ptr == Search_object::spat_configE)
            obj.*to_ptr = 'E';
        else if(obj.*from_ptr == Search_object::spat_config3)
            obj.*to_ptr = '3';
        else
            obj.*to_ptr = 'X';
*/
}

// for each object, if its relevant properties are all known, and it is currently uncrowded, mark it as known_perceptual_properties
// the size() of the crowder_group unambiguously shows whether the object is crowded or not.
void Visual_model::update_crowding_status()
{
    for(auto& obj : search_objects) {
//    	if(obj.current_crowding) assert(obj.crowder_group.size() > 1);
//    	if(obj.current_crowding && obj.crowder_group.size() == 0) cout <<  obj.crowder_group.size() << ' ';
//        if(!obj.current_crowding) {
        if(obj.crowder_group.size() == 0) {
//        	assert(obj.crowder_group.size() == 0);
            switch(condition) {
                case Condition_e::CSF:
                    if(obj.perc_color != '.') {
                        obj.known_perceptual_properties = true;
                        }
                    break;
                case Condition_e::COC:
                    if(obj.perc_color != '.' && obj.perc_orientation != '.')
                        obj.known_perceptual_properties = true;
                    break;
                case Condition_e::SHP:
                    if(unitary_shape_c) {
                        if(obj.perc_shape != '.') {
                            obj.known_perceptual_properties = true;
                            }
                        }
                    else {
                        if(all_of(obj.perc_spat_config.begin(), obj.perc_spat_config.end(), [](const char c) {return c != '.';})) {
                            obj.known_perceptual_properties = true;
                            }
                        }
                    break;
                default:
                    assert(!"unknown condition in update_crowding_status");
                    break;
                }
            if(detailed_visual_trace_output && obj.known_perceptual_properties) {
                cout << "Visual: " << obj.ID << " now has known perceptual properties " << endl;
                }
            } // end of currently uncrowded object conditional
        } // end of each object loop
}


// update_Shape_information_gain
// For each object obj:
// Calculate the expected number of objects whose blank shape will become known
// if obj is fixated -- included in this calculation is obj as well as the surrounding
// objects. If an object's Shape property is already known (i.e. perc_shape != '.'),
// then it is not included in the expected value, so this is the "gain" in information
// that is expected by fixating obj. Since obj is expected to have eccentricity = 0
// after the move, if its shape is currently known, it will contribute zero, but if
// currently unknown, will contribute approximately 1.0 to the information_gain score
void Visual_model::update_Shape_information_gain()
{
    assert(unitary_shape_c);    // this function not implemented for non-unitary shape property

    // zero the scores first
    for(auto& obj : search_objects) {
        obj.Shape_information_gain = 0.0;
        }

    for(auto& obj : search_objects) {
        // credit this object obj with information gain if perceptual Shape is unknown
        if(obj.perc_shape == '.')
            // Expected value that Shape will be available assuming 0.0 eccentricity if fixated
            obj.Shape_information_gain += LinearConstSD_detection_probability(0.0, obj.size, common_intercept_c, shape_slope_g, common_sd_c);
        // for every other object whose shape is unknown, determine expected value that its Shape will be available if obj is fixated
        for(auto& other_obj : search_objects) {
            if(other_obj.ID == obj.ID)
                continue;
            if(other_obj.perc_shape != '.') // skip if shape is known
                continue;
            double eccentricity = cartesian_distance(obj.location, other_obj.location);
            obj.Shape_information_gain += LinearConstSD_detection_probability(eccentricity, other_obj.size, common_intercept_c, shape_slope_g, common_sd_c);
            }
        }
}


// update_Feature_information_gain
// For each object obj:
// Calculate the expected number of objects whose blank features will become known
// if obj is fixated -- included in this calculation is obj as well as the surrounding
// objects. The score is calculated over both color and shape so that e.g. twice as much
// credit is given if both features would become available instead of just one.
// If an object's color property is already known (i.e. perc_color != '.'),
// then it is not included in the expected value, so this is the "gain" in information
// that is expected by fixating obj. Since obj is expected to have eccentricity = 0
// after the move, if its color is currently known, it will contribute zero, but if
// currently unknown, will contribute approximately 1.0 to the information_gain score
// Likewise, the orientation propety is computed for each object as well, and added to the information_gain score
void Visual_model::update_Feature_information_gain()
{
    assert(unitary_shape_c);    // this function not implemented for non-unitary shape property

    // zero the scores first
    for(auto& obj : search_objects) {
        obj.Feature_information_gain = 0.0;
        }

    for(auto& obj : search_objects) {
        // credit this object obj with information gain if perceptual Color is unknown
        if(obj.perc_color == '.') {
            // Expected value that Color will be available assuming 0.0 eccentricity if fixated
            obj.Feature_information_gain += LinearConstSD_detection_probability(0.0, obj.size, common_intercept_c, color_slope_g, common_sd_c);
            }
        // ditto for Orientation
        if(obj.perc_orientation == '.') {
            // Expected value that Orientation will be available assuming 0.0 eccentricity if fixated
            obj.Feature_information_gain += LinearConstSD_detection_probability(0.0, obj.size, common_intercept_c, orientation_slope_g, common_sd_c);
            }
        // for every other object whose Color is unknown, determine expected value that its Color will be available if obj is fixated
        // and ditto for Orientation
        for(auto& other_obj : search_objects) {
            if(other_obj.ID == obj.ID) // skip if same object
                continue;
            // we only need to compute eccentricity if we are going to need it, tweak later if useful
            if(other_obj.perc_color == '.') {// if color is unknown
                double eccentricity = cartesian_distance(obj.location, other_obj.location);
                obj.Feature_information_gain += LinearConstSD_detection_probability(eccentricity, other_obj.size, common_intercept_c, color_slope_g, common_sd_c);
                }
             if(other_obj.perc_orientation == '.') {// if orientation is unknown
                double eccentricity = cartesian_distance(obj.location, other_obj.location);
                obj.Feature_information_gain += LinearConstSD_detection_probability(eccentricity, other_obj.size, common_intercept_c, orientation_slope_g, common_sd_c);
                }
           }
        }
}



void Visual_model::check_target_status(Condition_e condition, const Search_objects_t& search_objects)
{
    switch(condition) {
        case Condition_e::CSF:
            check_CSF_target_status(search_objects);
            break;
        case Condition_e::COC:
            check_COC_target_status(search_objects);
            break;
        case Condition_e::SHP:
            check_SHP_target_status(search_objects);
            break;
        default:
            assert(!"unknown condition in check_target_status");
            break;
        }
}
//target_status_trace_output
void Visual_model::check_CSF_target_status(const Search_objects_t& search_objects)
{
//    bool target_sens_property_present = false;
//    bool target_perc_property_present = false;
    for(const auto& obj : search_objects) {
		if(obj.is_target) {
			if(obj.perc_color == 'G') n_target_illusory_distractors++;
			else if(obj.sens_color == 'R' && obj.perc_color == '.')  n_target_illusory_blanks++;
			}
		else  { // not a target
			if(obj.perc_color == 'R') n_illusory_targets++;
			}
//        if(obj.sens_color == 'R')
//            target_sens_property_present = true;
//        if(obj.perc_color == 'R')
//            target_perc_property_present = true;
        if(target_status_trace_output) {
			if(obj.is_target && obj.sens_color == '.' && obj.perc_color == '.')
				cout << trial_g << ' ' << n_fixations << " ---CSF " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target color is neither available nor visible" << endl;
			if(obj.is_target && obj.sens_color == '.' && obj.perc_color == 'R')
				cout << trial_g << ' ' << n_fixations << " ---CSF " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target color is not available but is visible" << endl;
	// 		if(obj.is_target && obj.sens_color == 'R' && obj.perc_color != 'R')
	//            cout << trial_g << ' ' << n_fixations << " ---CSF " << obj.name << ' ' << obj.ID << 				' ' << obj.eccentricity << " target color is available but not visible" << endl;
	//       if(obj.is_target && obj.perc_color == 'R')
	//            cout << trial_g << ' ' << n_fixations << " +++CSF " << obj.name << ' ' << obj.ID << 				' ' << obj.eccentricity << " target visible" << endl;
			if(obj.is_target && obj.sens_color == 'R' && obj.perc_color == '.')
				cout << trial_g << ' ' << n_fixations << " ???CSF " << obj.name << ' ' << obj.ID <<  " target color available but blank is visible" << endl;
			if(obj.is_target && obj.sens_color == 'R' && obj.perc_color == 'G')
				cout << trial_g << ' ' << n_fixations << " ???CSF " << obj.name << ' ' << obj.ID <<  " target color available but target looks like illusory distractor" << endl;
			if(obj.is_target && obj.sens_color == '.' && obj.perc_color == 'G')
				cout << trial_g << ' ' << n_fixations << " ---CSF " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target color not available but target looks like illusory distractor" << endl;
			if(!obj.is_target && obj.perc_color == 'R')
				cout << trial_g << ' ' << n_fixations << " !!!CSF " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " illusory target visible" << endl;
			}
		}
/*	if(target_sens_property_present) {
		n_target_sens_property_present++;
		}
    if(target_perc_property_present) {
		n_target_perc_property_present++;
		}
    if(target_sens_property_present && !target_perc_property_present) {
		n_target_property_overwritten++;
		}
*/
}

void Visual_model::check_COC_target_status(const Search_objects_t& search_objects)
{
    for(const auto& obj : search_objects) {
		if(obj.is_target) {
			if ((obj.perc_color == 'G' && obj.perc_orientation == 'V') ||
				(obj.perc_color == 'R' && obj.perc_orientation == 'H')) {
					n_target_illusory_distractors++;
				}
			else if ((obj.sens_color == 'R' && obj.perc_color == '.') ||
					 (obj.sens_orientation == 'V' && obj.perc_orientation == '.')) {
					n_target_illusory_blanks++;
				}
			}
		else { // not a target
			if(obj.perc_color == 'R' && obj.perc_orientation == 'V') {
				n_illusory_targets++;
				}
			}
	   if(target_status_trace_output) {
			if(obj.is_target && obj.perc_color == 'R' && obj.perc_orientation == 'V')
				cout << trial_g << ' ' << n_fixations << " +++COC " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target visible" << endl;
			if(obj.is_target && ((obj.sens_color == 'R' && obj.perc_color != 'R') ||
								(obj.sens_orientation == 'V' && obj.perc_orientation != 'V')))
				cout << trial_g << ' ' << n_fixations << " ---COC " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target masked" << endl;
			if(!obj.is_target && obj.perc_color == 'R' && obj.perc_orientation == 'V')
				cout << trial_g << ' ' << n_fixations << " !!!COC " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " illusory target visible" << endl;
			}
		}
}

void Visual_model::check_SHP_target_status(const Search_objects_t& search_objects)
{
    bool target_masked = false;
    bool illusory_target_visible = false;
    bool all_are_distractors = true;
    for(const auto& obj : search_objects) {
		if(obj.is_target) {
			if(obj.perc_shape == '5') n_target_illusory_distractors++;
			else if(obj.sens_shape == '2' && obj.perc_shape == '.') n_target_illusory_blanks++;
			}
		else { // not a target
			if(obj.perc_shape == '2') n_illusory_targets++;
			}
		if(target_status_trace_output) {
			if(obj.perc_shape != '5')
				all_are_distractors = false;
			if(obj.is_target && obj.perc_shape == '2')
				cout << trial_g << ' ' << n_fixations << " +++SHP " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target visible" << endl;
			if(obj.is_target && obj.sens_shape == '2' && obj.perc_shape == '5') {
				target_masked = true;
				cout << trial_g << ' ' << n_fixations << " ---SHP " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target shape sensed but overwritten with distractor shape" << endl;
				}
			if(obj.is_target && obj.sens_shape == '.' && obj.perc_shape == '5') {
				target_masked = true;
				cout << trial_g << ' ' << n_fixations << " ---SHP " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " target shape unsensed but overwritten with distractor shape" << endl;
				}
			if(!obj.is_target && obj.perc_shape == '2') {
				illusory_target_visible = true;
				cout << trial_g << ' ' << n_fixations << " !!!SHP " << obj.name << ' ' << obj.ID << ' ' << obj.eccentricity << " illusory target visible" << endl;
				}
			}
		}
		if(target_status_trace_output && all_are_distractors)
        cout << trial_g << ' ' << n_fixations << " ???SHP All objects appear to be distractors" << endl;
 //   if(target_masked && !illusory_target_visible)
//        cout << trial_g << ' ' << n_fixations << " !!!SHP target feature was overwritten" << endl;

}

// Calculate the Normal CDF value for x using mean m and sd s
// using the formula in terms of the <cmath> erf function
// see Wikipedia, Normal Distribution, for this equivalence
// with the cmath>
// this gives the probability that a random deviate will be <= x
double normal_cdf(double x, double mean, double sd)
{
    return (
    0.5 * (1 + erf((x - mean)/ (sd * sqrt(2.0))))
    );
}

bool LinearConstSD_available(double eccentricity, GU::Size size, double a, double b, double sd)
{
    // use the average of horizontal and vertical size
    double obj_size = (size.h + size.v) / 2.0;
    double mean = a + b * eccentricity;
    return lapsed_gaussian_detection_function(obj_size, mean, sd, 0.); // lapse_prob = 0;
}

double LinearConstSD_detection_probability(double eccentricity, GU::Size size, double a, double b, double sd)
{
    // use the average of horizontal and vertical size
    double obj_size = (size.h + size.v) / 2.0;
    double mean = a + b * eccentricity;
    //return lapsed_gaussian_detection_function(obj_size, mean, sd, 0.); // lapse_prob = 0;
    return normal_cdf(obj_size, mean, sd);
}


