//
//  Strategy9c.cpp
//  SimpleCrowdingModel
//
//  Created by David Kieras on 4/9/18.
//  Copyright © 2018 University of Michigan. All rights reserved.
//
// revised 7/13/19 to include recording of mean error RT

#include "Strategy.h"
#include "Display_creator.h"
//#include "Visual_modelV2a.h"
#include "Visual_modelV2e.h"
#include "Trial_statistics.h"
//#include "EPICLib/Random_utilities.h"
#include "Random_utilities.h"


#include <iostream>
#include <iomanip>
#include <sstream>
#include <set>
#include <cassert>

using namespace std;

// implements unified parameterized strategy pre-5a, 6b, 7
// version 9c allows search objects to be reordered by Visual_model
string Strategy::get_strategy_id_string()
{
    ostringstream oss;
    oss << "Strategy 9c with";
//    oss << "Strategy 9b look at furthest with";
    if(!confirm_positive_c && !confirm_negative_c)
        oss << " no confirmation";
    else {
        oss << " confirm";
        if(confirm_positive_c && confirm_negative_c)
            oss << " positive & negative";
        else if(confirm_positive_c)
            oss << " positive";
        else if(confirm_negative_c)
            oss << " negative";
        }
    if(max_n_fixations_g == 0)
        oss << ", eyes fixed";
    else if(max_n_fixations_g > 0 && max_n_fixations_g < 99)
 //       oss << ", respond absent if no target present within " << max_n_fixations_g << " fixations";
        oss << ", respond absent if no target present within " << calculate_average_max_n_fixations() << " fixations";
    else
        oss << ", search until target found";
    if(too_many_illusory_targets_c < 99)
        oss << ", or if " << too_many_illusory_targets_c << " illusory targets found";
    oss << "\n";
    if(confirm_positive_c)
        oss << "Skip confirm positive if fixation within " << close_enough_to_bypass_confirm_positive_c << " DVA of apparent target. ";
    if(confirm_negative_c)
    oss << "Skip confirm negative if at least " << enough_fixations_to_bypass_confirm_negative_c << " fixations already made. ";
    oss << "\n";

    return oss.str();
}


// input is time in ms, output is that time rounded up to next cycle start
double time_at_next_cycle_start(double time);


/* implements unified parameterized strategy pre-5a, 6b, 7 */
void Strategy::do_trial()
{
    // flip a coin to decide whether to use max_n_fixations_g + 0 or + 1, but only if max_n_fixations_g < 99
    int trial_max_n_fixations = max_n_fixations_g;
    if(max_n_fixations_mixture_probability_c > 0.0 && max_n_fixations_g < 99 && biased_coin_flip(max_n_fixations_mixture_probability_c))
        trial_max_n_fixations = max_n_fixations_g + 1;

    // bool target_nominated = false; // used to be here, moved to top of search loop 4/26/19
    
    n_repeat_fixations = 0;
    n_target_fixations = 0;
    fixated_objects.clear(); // keeps track of which objects have been fixated

    int n_illusory_target_fixations = 0;
    double stimulus_onset_time = uniform_random_variable(500., 50.); // start trial at a jittered time in range 450 - 550
    double time = stimulus_onset_time; // time for execution from stimulus presentation to response
    double motor_response_time = 0; // time for motor response
    double total_time = 0.;
    
    search_objects = display_creator.create_display(condition, set_size, polarity);

    Visual_model visual_model(condition, search_objects);
    // time allowance for initial registration in sensory store
    time = time + visual_delay_time_g;
    // round up to next rule execution cycle
    time = time_at_next_cycle_start(time);
    time += rule_execution_time_c; // for startup rule execution


    while(true) { // start of search loop
    if(strategy_trace_output) {
        cout << "Fixation: " << visual_model.get_n_fixations() << endl;
        }

    // 2/3/20 Visual_model can reorder the objects for crowding computation
    GU::Point eye_location = visual_model.get_eye_location();
    
    // this sort order is used to order nominated objects, since linear search will find the first match
    // as of 2/3/20 this ordering of nominations is closest object first -- least to greatest eccentricity from current eye location.
    sort(search_objects.begin(), search_objects.end(),
        [eye_location](const auto& obj1, const auto& obj2){return(GU::cartesian_distance(obj1.location, eye_location) < GU::cartesian_distance(obj2.location, eye_location));});
//        [eye_location](const auto& obj1, const auto& obj2){return(GU::cartesian_distance(obj1.location, eye_location) > GU::cartesian_distance(obj2.location, eye_location));});

    if(strategy_trace_output && false) {
        cout << trial_g << " Strategy: objects ordered by eccentricity:\n";
        for(const auto& obj : search_objects)
            cout << obj << endl;
        }
    
//    Search_objects_t::iterator nominated_object_it;
    //  initialize this here to reset on every loop iteration
    bool target_nominated = false;

    
    // NOMINATION PHASE nominate next object to look at; either a target or the closest object with unknown properties
    // all objects nominated are the closest for the type of the nomination because the contained has been sorted by eccentricity
    // The nomination results in an iterator, nominated_object_it, being set to either an apparent target, a possible target, or .end()
    switch(condition) {
        case Condition_e::CSF: {
            // object with target color -- apparent target
            if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return obj.perc_color == 'R';} ))
                    != search_objects.end()) {
                target_nominated = true;
                break;
                }
            // object with unknown color -- a possible target
            else if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                        [](const auto& obj){return obj.perc_color == '.';} ))
                        != search_objects.end()) {
                break;
                }
            else {
                assert(nominated_object_it == search_objects.end());
                break;
                }
            }
        case Condition_e::SHP: {
            // object with target shape - an apparent target
            if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return obj.perc_shape == '2';} ))
                    != search_objects.end()) {
                target_nominated = true;
                break;
                }
            // object with unknown shape or mis-recognized shape -- a possible target
            else if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
 //                   [](const auto& obj){return (obj.perc_shape == '.' || (obj.perc_shape != '2' && obj.perc_shape != '5'));} ))
                    [](const auto& obj){return (obj.perc_shape != '2' && obj.perc_shape != '5');} ))
                    != search_objects.end()) {
                break;
                }
             else {
                assert(nominated_object_it == search_objects.end());
                break;
                }
           }
        case Condition_e::COC: {
            // object with both target color and target orientation -- an apparent target
            if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return (obj.perc_color == 'R') && (obj.perc_orientation == 'V');} ))
                    != search_objects.end()) {
                target_nominated = true;
                break;
                }
            // object with target color but unknown orientation  -- a possible target
            else if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return (obj.perc_color == 'R') && (obj.perc_orientation == '.');} ))
                    != search_objects.end()) {
                 break;
                 }
            // object with unknown color but target orientation  -- a possible target
            else if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return (obj.perc_color == '.') && (obj.perc_orientation == 'V');} ))
                    != search_objects.end()) {
                break;
                }
            // object with both unknown color and unknown orientation  -- a possible target
            else if((nominated_object_it = find_if(search_objects.begin(), search_objects.end(),
                    [](const auto& obj){return (obj.perc_color == '.') && (obj.perc_orientation == '.');} ))
                    != search_objects.end()) {
                break;
                }
            else {
                assert(nominated_object_it == search_objects.end());
                break;
                }
            }
        }
     
    // even if no nominations made, it takes a cycle
    time += rule_execution_time_c; // for nomination rules Nominate_color_unknown_CSF, Nominate_match_color_CSF, etc

    // CHOOSE PHASE - act on the nominations and number of fixations

    // at least one choice rule always fires and takes a cycle
    time += rule_execution_time_c;    // for "lookat" or "response" rule
    
    // DECIDE WHAT TO DO NEXT

    // if no search fixations allowed (eyes fixed strategy) then respond depending on target_nominated and terminate the trial
    if(trial_max_n_fixations == 0) {
        if(target_nominated) {
            motor_response_time = make_positive_response();
            }
        else {
            motor_response_time = make_negative_response();
            }
        break; // out of search loop
        }
    assert(trial_max_n_fixations > 0);

    // if no nominated next_obj, we need to consider a negative response
    if(nominated_object_it == search_objects.end()) {
        // if not confirming negative responses, make the negative response and terminate the trial
        if(!confirm_negative_c) {
            motor_response_time = make_negative_response();
            break; // out of search loop
            }
        // check to see if maximum number of fixations have already been made; if so, respond negative
        if(visual_model.get_n_fixations() >= trial_max_n_fixations) {
            motor_response_time = make_negative_response();
            break; // out of search loop
            }

        // if confirming negative, consider doing a check eye movement
        // restore these special cases 7/13/18
        // go ahead and respond absent if we have made at least one fixation
        if(visual_model.get_n_fixations() >= enough_fixations_to_bypass_confirm_negative_c) {
            motor_response_time = make_negative_response();
            break; // out of search loop
            }

        // do confirmation: move eye to furthest object and check again
        // and either continue the search loop, or make a negative response
        else {
    	    // objects are in order of increasing eccentricity, so last object is furthest away
		    auto& object_to_fixate = search_objects.back();

/*            // objects are in order of decreasing eccentricity, so first object is furthest away
            auto& next_obj = search_objects.front();
*/
//            int saved_object_ID = next_obj.ID;
            if(strategy_trace_output) {
                cout << trial_g << " Strategy: Confirm-Negative look at: " << object_to_fixate << endl;
                }

            double eye_movement_time = move_eye_to_object(visual_model, object_to_fixate);
            time += eye_movement_time;
            // round up to start of next cycle
            time = time_at_next_cycle_start(time);
            time += rule_execution_time_c;    // for eyemovement_done rule execution

            // confirm that it is not a target
            auto& fixated_object = *fixated_object_it;
            bool non_target_confirmed = false;
            switch(condition) {
                case Condition_e::CSF: {
                    if(fixated_object.perc_color == 'G') {
                        non_target_confirmed = true;
                        }
                    break;
                    }
                case Condition_e::SHP: {
                    if(fixated_object.perc_shape == '5') {
                        non_target_confirmed = true;
                        }
                    break;
                    }
                case Condition_e::COC: {
                    if(!((fixated_object.perc_color == 'R') && (fixated_object.perc_orientation == 'V'))) {
                        non_target_confirmed = true;
                        }
                    break;
                    }
                }
            if(non_target_confirmed) {
                motor_response_time = make_negative_response();
                break; // out of search loop - trial is done
                }
            else {
                // NOTE NOTE NOTE ??? if it actually is a target, we waste a nomination cycle!
                // non-target was not confirmed, continue the loop
                continue; // continue search loop
                }
            } // end of case where no fixations made yet        } // end of no-nomination case
        } // end of case where no objects nominated
        
    // here only if an object was nominated:
    assert(nominated_object_it != search_objects.end());
    auto&  nominated_object = *nominated_object_it;
 //   int fixated_object_ID = nominated_object.ID;

    // if a target was not nominated, move the eye, and continue the loop.
    if(!target_nominated) {
        // check to see if maximum number of fixations have already been made; if so, respond negative
        if(visual_model.get_n_fixations() >= trial_max_n_fixations) {
            motor_response_time = make_negative_response();
            break; // out of search loop
            }
        if(strategy_trace_output) {
            cout << trial_g << " Strategy: look at possible target: " << nominated_object << endl;
            }
        double eye_movement_time = move_eye_to_object(visual_model, nominated_object);
        time += eye_movement_time;
        // round up to start of next cycle
        time = time_at_next_cycle_start(time);
        time += rule_execution_time_c;    // for eyemovement_done rule execution
        continue; // continue the search loop
        }
    // if a target was nominated, if confirmation needed, check it out and respond positive or continue the search loop to get a new nomination
    if(target_nominated) {
        // if no confirmation required, go ahead and make the response
        // debug 2/3/20
 //       cout << " *** TARGET NOMINATED: " << nominated_object.name << ' ' << nominated_object.ID << endl;
//        for(const auto& obj : search_objects)
 //           cout << obj << endl;

        if(!confirm_positive_c) {
            motor_response_time = make_positive_response();
            break; // out of search loop - trial is done
            }
        if(nominated_object.eccentricity <= close_enough_to_bypass_confirm_positive_c) {
            // debug 2/3/20
//            cout << " *** CONFIRMATION BYPASSED: " << nominated_object.name << ' ' << nominated_object.ID << endl;
//            for(const auto& obj : search_objects)
//                cout << obj << endl;
            motor_response_time = make_positive_response();
            break; // out of search loop - trial is done
            }
        // else move eye to the nominated object and confirm, but only if more fixations allowed
        // if more fixations are not allowed, respond negative, because target has not in fact been found. 190221
        if(visual_model.get_n_fixations() >= trial_max_n_fixations) {
//            motor_response_time = make_positive_response();  // was in Strategy9
            motor_response_time = make_negative_response();
            break; // out of search loop
            }
        if(strategy_trace_output) {
            cout << trial_g << " Strategy: Confirm-Positive look at apparent target: " << nominated_object << endl;
            }
        // move eye to nominated object
        double eye_movement_time = move_eye_to_object(visual_model, nominated_object);
        time += eye_movement_time;
        // round up to next rule cycle time for confirmation/discomfirmation rule
        time = time_at_next_cycle_start(time);
        time += rule_execution_time_c;    // for confirmation/disconfirmation rule execution
        // fixated_object_it is the valid iterator pointing to the fixated object after processing done
        auto fixated_object = *fixated_object_it;
        // confirm that fixated_object now has target properties
        // debug 2/3/20 NOTE THAT WE DO NOT REORDER THE OBJECTS AT THIS POINT - THEY ARE NOW IN THE ORDER VM2 left them in!
        bool target_confirmed = false;
        switch(condition) {
            case Condition_e::CSF: {
                if(fixated_object.perc_color == 'R') {
                    target_confirmed = true;
                    }
                break;
                }
            case Condition_e::SHP: {
                if(fixated_object.perc_shape == '2') {
                    target_confirmed = true;
                    }
                break;
                }
            case Condition_e::COC: {
                if((fixated_object.perc_color == 'R') && (fixated_object.perc_orientation == 'V')) {
                    target_confirmed = true;
                    }
                break;
                }
            }
        if(target_confirmed) {
            // debug 2/3/20
//            cout << " *** TARGET CONFIRMED: " << fixated_object.name << ' ' << fixated_object.ID << endl;;
//            for(const auto& obj : search_objects)
//                cout << obj << endl;
            motor_response_time = make_positive_response();
            break; // out of search loop - trial is done
            }
        else {
            // target was not confirmed, continue the loop
            // debug 2/3/20
            if(strategy_trace_output) cout << trial_g << " Strategy: apparent target " << fixated_object.name << ' ' << fixated_object.ID << " was not a target" << endl;
//            cout << "Strategy: apparent target " << next_obj.name << ' ' << next_obj.ID << " was not a target" << endl;
            n_illusory_target_fixations++;
            // ??? DK 3/16/19 stop and respond negative if we have seen too many illusory targets
            if(n_illusory_target_fixations >= too_many_illusory_targets_c) {
                motor_response_time = make_negative_response();
                break;  // out of search loop - trial is done
                }
            target_nominated = false; // looks like only place where target_nominated would have to be reset 4/26/19
            continue; // continue search loop
            }
        } // end of target nominated case
    } // end of search loop
    assert(time > 0);
    total_time = (time - stimulus_onset_time) + motor_response_time;
    assert(total_time > 0);

	if(strategy_trace_output || false)
		cout << trial_g << " Strategy: Times: pure visual delay: " << visual_delay_time_g // included in rule time total
		<< ", rules: " << time
		<< ", motor: " << motor_response_time
        << ", total: " << total_time
		<< ", n_fixations: " << visual_model.get_n_fixations()
        << ", n_target_illusory_distractors: " << visual_model.get_n_target_illusory_distractors()
        << ", n_target_illusory_blanks: " << visual_model.get_n_target_illusory_blanks()
		<< ", n_illusory_targets: " << visual_model.get_n_illusory_targets()
//		<< ' ' << ", n_target_sens_property_present: " << visual_model.get_n_target_sens_property_present()
//		<< ' ' << ", n_target_perc_property_present: " << visual_model.get_n_target_perc_property_present()
//        << ", n_target_property_overwritten: " << visual_model.get_n_target_property_overwritten()
		<< endl;

	
    // response has been made, accumulate statistics
    // accumulate statistics
    if(!error) {
        statistics.mean_RT.update(total_time);
        statistics.mean_n_fixations.update(visual_model.get_n_fixations());
        }
	else {
		statistics.mean_error_RT.update(total_time);
        statistics.mean_error_n_fixations.update(visual_model.get_n_fixations());
		}
    // average illusory target regardless of positive vs negative trials ??? DK 3/13/19 ??? correct only would be more useful here - should be impossible in CSF-
    statistics.mean_illusory_target_fixations.update(n_illusory_target_fixations);
    statistics.mean_repeat_fixations.update(n_repeat_fixations);
    statistics.mean_target_fixations.update(n_target_fixations);
    // calculate how many times the display was processed, including intial onset - counts the initial central fixation
    if(strategy_trace_output)
		cout << visual_model.get_n_fixations()
			<< ' ' << visual_model.get_n_target_illusory_distractors()
			<< ' ' << visual_model.get_n_target_illusory_blanks()
			<< ' ' << visual_model.get_n_illusory_targets()
//			<< ' ' << visual_model.get_n_target_sens_property_present()
//			<< ' ' << visual_model.get_n_target_perc_property_present()
//			<< ' ' << visual_model.get_n_target_property_overwritten()
			<< endl;
    double n_process_display = 1 + visual_model.get_n_fixations();
    statistics.mean_n_target_illusory_distractor.update(visual_model.get_n_target_illusory_distractors() / n_process_display);
    statistics.mean_n_target_illusory_blank.update(visual_model.get_n_target_illusory_blanks() / n_process_display);
    statistics.mean_n_illusory_targets.update(visual_model.get_n_illusory_targets() / n_process_display);
//    statistics.mean_n_target_sens_property_present.update(visual_model.get_n_target_sens_property_present() / n_process_display);
//    statistics.mean_n_target_perc_property_present.update(visual_model.get_n_target_perc_property_present() / n_process_display);
//    statistics.mean_n_target_property_overwritten(visual_model.get_n_target_property_overwritten() / n_process_display);
    return;
} // end of do_trial


double Strategy::move_eye_to_object(Visual_model& visual_model, Search_object& object_to_fixate)
{
    int fixated_object_ID = object_to_fixate.ID; // save ID for later, since iterators may become invalid
    // move eye - nominated_object_it is pointing to next object to look at
    double eye_movement_time = visual_model.move_eye(object_to_fixate);
    // Visual_model may reorder the objects, so the iterator is now invalid
    // reset it using the saved ID.
    fixated_object_it = find_if(search_objects.begin(), search_objects.end(),
        [fixated_object_ID] (Search_object& obj) {return obj.ID == fixated_object_ID;});
    assert(fixated_object_it != search_objects.end());
    Search_object& fixated_obj = *fixated_object_it;
    if(fixated_obj.is_target) // it might be the target we are looking at
        n_target_fixations++;
    auto result = fixated_objects.insert(fixated_object_ID);
    if(!result.second) // ID was not inserted because already there
        n_repeat_fixations++;
    return eye_movement_time;
}

// return the response time
double Strategy::make_positive_response()
{
    bool processing_error = (polarity == Polarity_e::NEGATIVE); // error if correct response was opposite to positive
    bool oops_error = biased_coin_flip(positive_baseline_error_rate_c);
    error = processing_error || oops_error; // error rate is at least guess_error rate;
    response = (!error) ? true : false; // if no error, response is yes - true -; opposite if error
    // debug 02/03/20
    if(strategy_trace_output) {
        cout << trial_g << " Strategy: target present";
        if(processing_error)
            cout << " *** FALSE ALARM ERROR";
        else if(oops_error)
            cout << " *** OOPS FALSE ALARM ERROR";
        cout  << endl;
        }
    statistics.prop_errors.update(error);
//        statistics.prop_illusory_target_fixations.update(false); //??? What is baseline for proportion denominator?
    double motor_response_time = (response == last_response) ? same_key_motor_response_time_c : different_key_motor_response_time_c;
    last_response = response;
    return motor_response_time;
}

double Strategy::make_negative_response()
{
    bool processing_error = (polarity == Polarity_e::POSITIVE); // error if correct response was opposite to positive
    bool oops_error = biased_coin_flip(negative_baseline_error_rate_c);
    error = processing_error || oops_error; // error rate is at least guess_error rate;
    response = (!error) ? false : true; // if no error, response is no - false -; opposite if error
    if(strategy_trace_output) {
        cout << trial_g << " Strategy: target absent";
       if(processing_error)
            cout << " *** MISS ERROR";
        else if(oops_error)
            cout << " *** OOPS MISS ERROR";
        cout << endl;
        }
    statistics.prop_errors.update(error);
    double motor_response_time = (response == last_response) ? same_key_motor_response_time_c : different_key_motor_response_time_c;
    last_response = response;
    return motor_response_time;
}

// input is time in ms, output is that time rounded up to next cycle start
double time_at_next_cycle_start(double time)
{
    int integral_part = int(time);
    int n_cycles = integral_part / rule_execution_time_c;
    double result = (n_cycles + 1) * rule_execution_time_c;
    assert(result > 0.);
    return result;
}
