//
//  Display_creator.cpp
//  
//
//  Created by David Kieras on 3/7/18.
//

/*
Wolfe, Palmer, Horowitz 2010 model
Conditions:
Number of objects: 3, 6, 12, 18
Cue condition: Color Single Feature (SCF), Color Orientation Conjunction (COC), Shape (SHP)
Trial Polarity: Positive (Cued target is present), Negative (cued target is absent).
50% Positive, 50% Negative

Stimuli:
On each trial, positive/negative chosen at random, number of objects chosen at random

CSF: vertical bars 1 X 3.5; target: red vertical bar; distractors: green vertical bar

COC: vertical 1 X 3.5 or horizontal 3.5 X 1 bars; target: red vertical; distractors: red horizontal, green vertical.

SHP: "digital" 2s and 5s 1.5 X 2.7; target: 2, distractors 5

Visual field is about 22.5 x 22.5 degrees, divided up into 4.5 X 4.5 degree squares
in 5 X 5 matrix.
Apparently objects were put into a randomly selected square with no more than one object is placed in a square.
An object within a square was put at a random location within the square (not further described).
This is location 'jitter'.

Say we will maintain at least 0.25 dva between objects then: gives 4 dva range both x and y center location to vary in

CSF: center +/- 1.5 x, +/- .25 y U(-1.5, + 1.5) + x, U(-.25, +.25) + y or
                                    uniform_random_variable(x, 1.5), (y, .25)
COC vertical bar, +/- 1.5 x, +/- .25 y
    horizontal bar +/- .25 x, +/- 1.5 y
SHP center +/- 1.25 x, .65 y

Trial sequence:

1. Central fixation point .7 X .7 dva. Subjects were told to keep eyes on fixation point
but eye movements were not monitored. Fixation point apparently remains - not clear

2. Beep, then 500 ms delay, display appears.

3. S presses 'z' or '/' for whether target is absent or present.

4. Display disappears.

5. Correct/error feedback fro 500 ms.

6. Wait ITI=1000 ms for next trial start.

Subject could pause experiment by pressing space bar

Stimulus construction:
Randomize locations, then put N distractors in;
flip coin for trial polarity;
if positive, replace first distractor with target (chosen from cue condition);
if negative, leave as is.

Data:
RT in each condition, count of number of eye movements after space bar
Dwell is actually not meaningful for first fixation (eye was placed at fixation point long ago)
or last fixation (response is underway, eye not moved until trial is over).
So fixations should not be counted until trial starts, and dwell statistics on them not computed until trial ends.


*/

#include "Display_creator.h"
#include "Search_object.h"
#include "Program_constants.h"
/*#include "EPICLib/Geometry.h"
#include "EPICLib/Numeric_utilities.h"
#include "EPICLib/Random_utilities.h"
#include "EPICLib/Symbol_utilities.h"
#include "EPICLib/Ostream_format_saver.h" */
#include "Geometry.h"
#include "Numeric_utilities.h"
#include "Random_utilities.h"
#include "Symbol_utilities.h"
#include "Ostream_format_saver.h"
#include <iostream>
#include <iomanip>
#include <string>
#include <sstream>
#include <algorithm>
#include <random>
#include <cassert>
#include <cmath>

namespace GU = Geometry_Utilities;
//using GU::Point;
using namespace std;

// handy internal constants
const Symbol Fixation_point_c("Fixation_point");
const Symbol Display_c("Display");
const Symbol Dummy_c("Dummy");
const Symbol Go_response_c("SP");
const Symbol Response_absent_c("z");
const Symbol Response_present_c("/");


const char Red_c{'R'};
const char Green_c{'G'};
const char Black_c{'B'};
const char Vertical_c{'V'};
const char Horizontal_c{'H'};
const char Rectangle_c{'r'};
const char Digital_2_c{'2'};
const char Digital_5_c{'5'};

const double CSF_x_jitter = 1.5;
const double CSF_y_jitter = .25;
const double SHP_x_jitter = 1.25;
const double SHP_y_jitter = .65;
const double COCH_x_jitter = .25;
const double COCH_y_jitter = 1.5;
const double COCV_x_jitter = 1.5;
const double COCV_y_jitter = .25;

GU::Point jitter(GU::Point point, double x_deviation, double y_deviation);


Display_creator::Display_creator()
{
//Visual field is about 22.5 x 22.5 degrees, divided up into 4.5 X 4.5 degree squares
//in 5 X 5 matrix.
    // fill the location vector 5 X 5 grid, skip the center position
    // locations are centers of 4.5 x 4.5 boxes; skip the center position
    // maybe draw a box?
    for(int x = -2; x <= +2; x++) {
        for(int y = -2; y <= +2; y++) {
            if(x == 0 && y == 0)
                continue;
            all_locations.push_back(GU::Point(x * 4.5, y * 4.5));
            }
        }
    
    initialize();

}



// called externally to start up device
void Display_creator::initialize()
{
    // reset accumulators
    whole_display_properties.reset();
    initial_display_properties.reset();

}


std::vector<Search_object> Display_creator::create_display(Condition_e condition_, int set_size, Polarity_e polarity_)
{
    condition = condition_;
    n_objects = set_size; // to reuse code and improve modularity
    polarity = polarity_;
    
    //Was done>TO DO: replace with modern std::shuffle
    // get a randomization of the possible locations
//    random_shuffle(all_locations.begin(), all_locations.end(), random_int);
    shuffle(all_locations.begin(), all_locations.end(), get_Random_engine());

    // empty the container of search objects
    search_objects.clear();
    
    switch (condition) {
        case Condition_e::CSF: generate_CSF_stimulus(); break;
        case Condition_e::COC: generate_COC_stimulus(); break;
        case Condition_e::SHP: generate_SHP_stimulus(); break;
        };
    // add any essential metrics to the search objects
    calculate_display_metrics();
    // calculate and average the stimulus characteristics
    if(compute_and_output_display_statistics) {
        calculate_display_statistics();
        //initial_display_properties.output();
        }
    assert(search_objects.size() == n_objects);
    return search_objects;
}


// CSF: vertical bars 1 X 3.5; target: red R vertical V bar; distractors: green G vertical V bar
// object size is avg (1+3.5)/2 = 2.25
void Display_creator::generate_CSF_stimulus()
{
    // fill the container of search objects, using the first n_objects locations
    // fill container with distractors
    for(int i = 0; i < n_objects; i++) {
        search_objects.push_back(Search_object("DistGrnV", jitter(all_locations[i], CSF_x_jitter, CSF_y_jitter),
            GU::Size(1., 3.5), Green_c, Rectangle_c, Vertical_c));
        }
    // if the trial is positive polarity, replace the [0] distractor with the appropriate target
    if(polarity == Polarity_e::POSITIVE) {
        // create the target, keeping a separate copy for convenience
        target = Search_object("TargetRedV", jitter(all_locations[0], CSF_x_jitter, CSF_y_jitter),
            GU::Size(1., 3.5),Red_c, Rectangle_c, Vertical_c, true);
        search_objects[0] = target;
        }
}

// COC: vertical 1 X 3.5 or horizontal 3.5 X 1 bars; target: red vertical; distractors: red horizontal, green vertical.
// limitation of distractors to two instead of three possibilities is explicit in paper - P. 1306.
// Paper is not explicit about distribution of the two kinds of distractors, especially at n = 3.
// Assuming here that there should be at least one of each kind of distractor.
// object size is avg (1+3.5)/2 = 2.25
void Display_creator::generate_COC_stimulus()
{
    // fill the container of search objects, using the first n_objects locations
    // first fill with distractors, an equal number of each type,
    // then if a positive trial, the target object will be put into the first cell of the container.
    // so first fill container with distractors; since there are two distractor types, flip a coin to
    // decide which distractor fills the first half of the container, and which fills the second half.
    // This way, the target replaces an equal number of distractors of each type over trials.
    // n_objects == 3 is special case because it is odd - fill the first two cells with a distractor,
    // the third cell with the other distractor, ensuring that at least one distractor of each type is present.

    // create skeletons for each kind of distractor
    Search_object dist1("DistRedH", GU::Size(3.5, 1.), Red_c, Rectangle_c, Horizontal_c);
    double x_jitter1 = COCH_x_jitter;
    double y_jitter1 = COCH_y_jitter;
    Search_object dist2("DistGrnV", GU::Size(1., 3.5), Green_c, Rectangle_c, Vertical_c);
    double x_jitter2 = COCV_x_jitter;
    double y_jitter2 = COCV_y_jitter;
    // now flip a coin and swap the two skeletons to decide who goes first into the container
    if(biased_coin_flip(0.5)) {
        swap(dist1, dist2);
        swap(x_jitter1, x_jitter2);
        swap(y_jitter1, y_jitter2);
        }
    // n_objects is 3, 6, 12, or 18; the "extra" distractor has been chosen at random
    // for n=3, ensure that at least one distractor of each type is chosen by using two of the first kind, one of the other.
    int n_half_objects = (n_objects == 3) ? 2 : n_objects / 2;
    for(int i = 0; i < n_half_objects; i++) {
        dist1.set_location(jitter(all_locations[i], x_jitter1, y_jitter1));
        dist1.set_ID(); // create unique object ID
        search_objects.push_back(dist1); // copy it into the container.
        }
    for(int i = n_half_objects; i < n_objects; i++) {
        dist2.set_location(jitter(all_locations[i], x_jitter2, y_jitter2));
        dist2.set_ID(); // create unique object ID
        search_objects.push_back(dist2); // copy it into the container.
        }
/*
    for(int i = 0; i < n_half_objects; i++) {
        search_objects.push_back(Search_object("DistRedH", jitter(all_locations[i], COCH_x_jitter, COCH_y_jitter),
            GU::Size(3.5, 1.), Red_c, Rectangle_c, Horizontal_c));
        }
    for(int i = n_half_objects; i < n_objects; i++)
        search_objects.push_back(Search_object("DistGrnV", jitter(all_locations[i], COCV_x_jitter, COCV_y_jitter),
            GU::Size(1., 3.5), Green_c, Rectangle_c, Vertical_c));
*/
    // if the trial is positive polarity, replace the [0] distractor with the appropriate target
    if(polarity == Polarity_e::POSITIVE) {
        // create the target, keeping a separate copy for convenience
        target = Search_object("TargRedV", jitter(all_locations[0], COCV_x_jitter, COCV_y_jitter),
            GU::Size(1., 3.5), Red_c, Rectangle_c, Vertical_c, true);
        search_objects[0] = target;
        }
}

// SHP: "digital" 2s and 5s 1.5 X 2.7; target: 2, distractors 5
// object size is avg (1.5+2.7)/2 = 2.1

void Display_creator::generate_SHP_stimulus()
{
    // fill the container of search objects, using the first n_objects locations
    // fill container with distractors
    for(int i = 0; i < n_objects; i++) {
//        search_objects.push_back(Search_object("Dist5", jitter(all_locations[i], SHP_x_jitter, SHP_y_jitter),
//            GU::Size(1.5, 2.7), Black_c, Circle_c, Nil_c));
        search_objects.push_back(Search_object("Dist5", jitter(all_locations[i], SHP_x_jitter, SHP_y_jitter),
            GU::Size(1.5, 2.7), Black_c, Digital_5_c, Vertical_c));
        }
    // if the trial is positive polarity, replace the [0] distractor with the appropriate target
    if(polarity == Polarity_e::POSITIVE) {
        // create the target, keeping a separate copy for convenience
//        target = Search_object("Targ2", jitter(all_locations[0], SHP_x_jitter, SHP_y_jitter),
//            GU::Size(1.5, 2.7), Black_c, Square_c, Nil_c);
        target = Search_object("Targ2", jitter(all_locations[0], SHP_x_jitter, SHP_y_jitter),
            GU::Size(1.5, 2.7), Black_c, Digital_2_c, Vertical_c, true);
        search_objects[0] = target;
        }
}

void Display_properties_accumulator::output() const
{
    Ostream_format_saver ofs(cout);
    // Set output to show two decimal places for floating point numbers
//    cout << fixed << setprecision(2) << endl;
    cout.setf(ios::fixed, ios::floatfield);
    cout.precision(2);
    cout << eccentricities.get_mean() << '\t'
        << spacings.get_mean() << '\t'
        << numbers_of_crowding_flankers.get_mean() << '\t'
        << dbn_of_number_of_crowding_flankers.get_max() << '\t'
        << spacings_of_crowding_flankers.get_mean() << ":\t";
    auto dbn = dbn_of_number_of_crowding_flankers.get_distribution();
    for(auto x : dbn)
        cout << '\t' << x;
    cout << endl;
}

void Display_creator::calculate_display_metrics()
{
    // calculate the display metrics for every possible object i and other object j (!= i)
    for(int i = 0; i < search_objects.size(); i++) { // i is current object
        // find and save nearest neighbor distance
        double nearest_neighbor_distance = 99.;
        for(int j = 0; j < search_objects.size(); j++) { // j is possible other object
            if(j != i) { // j is other object
                double eccentricity = GU::cartesian_distance(search_objects[i].location, search_objects[j].location);
                if(eccentricity < nearest_neighbor_distance)
                    nearest_neighbor_distance = eccentricity;
                } // end of possible other object j
            } // end of possible other object loop
        search_objects[i].nearest_neighbor_distance = nearest_neighbor_distance;
        } // end of current object i
}

void Display_creator::reset_display_statistics()
{
    initial_display_properties.reset();
    whole_display_properties.reset(); // why reset here?
}

// let each object be the current object, calculate the eccentricity to each other object (target),
// and for that target object calculate the number of flanking objects within the critical spacing (Bouma)
// calculate similar statistics for eccentricity from fixation point
void Display_creator::calculate_display_statistics()
{
    initial_display_properties.n_trials++;
	initial_display_properties.number_of_objects.update(search_objects.size());
    whole_display_properties.n_trials++;
	whole_display_properties.number_of_objects.update(search_objects.size());

    // calculating the initial display statistics is a special case because the fixation point is not a search_object
    GU::Point current_fixation_location(0., 0.);
    for(int j = 0; j < search_objects.size(); j++) {
        // each object j is possible target for next fixation
        calculate_possible_next_fixation_statistics(current_fixation_location, j, initial_display_properties);
        }
    
    // calculate the display statistics for every possible current fixation i and next fixation j (!= i) and possible flanker k (!= j)
    for(int i = 0; i < search_objects.size(); i++) { // object i is current point of fixation
        GU::Point current_fixation_location = search_objects[i].location;
        for(int j = 0; j < search_objects.size(); j++) {
            if(j != i) { // object j is a possible target
                calculate_possible_next_fixation_statistics(current_fixation_location, j, whole_display_properties);
                } // end of possible target j
            } // end of possible target loop
        } // end of current fixation point i
}

void Display_creator::calculate_possible_next_fixation_statistics(GU::Point current_assumed_fixation_point, int possible_next_fixation_index, Display_properties_accumulator& properties_accumulator)
{
    GU::Point possible_next_fixation_location = search_objects[possible_next_fixation_index].location;
    double eccentricity = GU::cartesian_distance(current_assumed_fixation_point, possible_next_fixation_location);
    properties_accumulator.eccentricities.update(eccentricity);
    properties_accumulator.dbn_of_eccentricities.update(eccentricity);
    int n_crowders = 0; // how many flanker objects are crowding this possible target possible_next_fixation_index
    for(int k = 0; k < search_objects.size(); k++) {
        if(k != possible_next_fixation_index) { // k is a possible flanker of target possible_next_fixation_index (current point of fixation could be a flanker)
            double spacing = GU::cartesian_distance(possible_next_fixation_location, search_objects[k].location);
            properties_accumulator.spacings.update(spacing);
            if(spacing <= bouma_fraction_c * eccentricity) {
                // flanker k is crowding target j
                n_crowders++;
                // if this flanker is within crowding distance, update the average for that spacing
                properties_accumulator.spacings_of_crowding_flankers.update(spacing);
                } // end of crowding flanker
            } // end of possible flanker k
        }
    properties_accumulator.numbers_of_crowding_flankers.update(n_crowders);
    properties_accumulator.dbn_of_number_of_crowding_flankers.update(n_crowders);
    properties_accumulator.mean_n_crowders_for_eccentricity.update(eccentricity, n_crowders);
}


void Display_creator::output_display_statistics() const
{
    output_display_properties_accumulator("Initial display properties", initial_display_properties);
    output_display_properties_accumulator("Whole display properties", whole_display_properties);
}

void Display_creator::output_display_properties_accumulator(const char* message, const Display_properties_accumulator& acc) const
{
    cout << message << endl;
//	cout << "Ntrials" << '\t' << "NObjs" << '\t' <<
	cout << "Ntrials" << '\t' <<
		"NObMin" << '\t' << "NObM" << '\t' << "NObMx" << '\t' <<
		"EccMin" << '\t' << "EccM" << '\t' << "EccMx" << '\t' <<
		"SpaMin" << '\t' << "SpaM" << '\t' << "SpaMx" << '\t' <<
		"NCFMin" << '\t' << "NCFM" << '\t' << "NCFMx" << '\t' <<
		"SCFMin" << '\t' << "SCFM" << '\t' << "SCFMx" << endl;
	cout << acc.n_trials << '\t' <<
		acc.number_of_objects.get_min() << '\t' << acc.number_of_objects.get_mean() << '\t' << acc.number_of_objects.get_max() << '\t' <<
		acc.eccentricities.get_min() << '\t' << acc.eccentricities.get_mean() << '\t' << acc.eccentricities.get_max() << '\t' <<
		acc.spacings.get_min() << '\t' << acc.spacings.get_mean() << '\t' << acc.spacings.get_max() << '\t' <<
		acc.numbers_of_crowding_flankers.get_min() << '\t' << acc.numbers_of_crowding_flankers.get_mean() << '\t' << acc.numbers_of_crowding_flankers.get_max() << '\t' <<
		acc.spacings_of_crowding_flankers.get_min() << '\t' << acc.spacings_of_crowding_flankers.get_mean() << '\t' << acc.spacings_of_crowding_flankers.get_max() << endl;
	// output distributions
	cout << "EccDbn"  << '\t' << acc.dbn_of_eccentricities.get_min() << '\t' << acc.dbn_of_eccentricities.get_max() << '\t' << acc.dbn_of_eccentricities.get_n();
	for(int i = 0; i < acc.dbn_of_eccentricities.get_n_bins(); i++)
		cout << '\t' << acc.dbn_of_eccentricities[i];
	cout << endl;
	cout << "NCFDbn"  << '\t' << acc.dbn_of_number_of_crowding_flankers.get_min() << '\t' << acc.dbn_of_number_of_crowding_flankers.get_max() << '\t' << acc.dbn_of_number_of_crowding_flankers.get_n();
	for(int i = 0; i < acc.dbn_of_number_of_crowding_flankers.get_n_bins(); i++)
		cout << '\t' << acc.dbn_of_number_of_crowding_flankers[i];
	cout << endl;
	cout << "MNCEcc"  << '\t' << acc.mean_n_crowders_for_eccentricity.get_min() << '\t' << acc.mean_n_crowders_for_eccentricity.get_max() << '\t' << acc.mean_n_crowders_for_eccentricity.get_n();
	for(int i = 0; i < acc.mean_n_crowders_for_eccentricity.get_n_bins(); i++)
		cout << '\t' << acc.mean_n_crowders_for_eccentricity[i].get_mean();
	cout << endl;
}


GU::Point jitter(GU::Point point, double x_jitter, double y_jitter)
{
    return GU::Point(uniform_random_variable(point.x, x_jitter), uniform_random_variable(point.y, y_jitter));
}



