//
//  DISPLAY_CREATOR_H
//  
//
//  Created by David Kieras on 3/7/18.
//

#ifndef DISPLAY_CREATOR_H
#define DISPLAY_CREATOR_H

#include "Search_object.h"
#include "Statistics.h"
//#include "EPICLib/Statistics.h"

#include <vector>

// a class for accumulating display characteristics relevant to crowding
struct Display_properties_accumulator {
    int n_trials;
    Mean_accumulator number_of_objects;	// for validity checking
    Mean_accumulator eccentricities;
    Mean_accumulator spacings;
    Mean_accumulator numbers_of_crowding_flankers;
    Mean_accumulator spacings_of_crowding_flankers;
    static constexpr int n_crowding_flankers_dbn_bins_c = 10;
    static constexpr int eccentricity_n_bins_c = 32;
    static constexpr int eccentricity_bin_size_c = 1;
    Distribution_accumulator dbn_of_eccentricities{eccentricity_n_bins_c, eccentricity_bin_size_c};
    // 10 bins for zero through 9 flankers, more aren't counted in last bin but included in total number
    Discrete_distribution_accumulator dbn_of_number_of_crowding_flankers{n_crowding_flankers_dbn_bins_c};
    // 32 bins of 1 deg each for the average number of crowders at that eccentricity
    Binned_mean_accumulators mean_n_crowders_for_eccentricity{eccentricity_n_bins_c, eccentricity_bin_size_c};
    void reset()
        {
        n_trials = 0;
        number_of_objects.reset();
        eccentricities.reset();
        spacings.reset();
        numbers_of_crowding_flankers.reset();
        spacings_of_crowding_flankers.reset();
        dbn_of_eccentricities.reset();
        dbn_of_number_of_crowding_flankers.reset();
        mean_n_crowders_for_eccentricity.reset();
        }
    // update using means and counts from another accumulator of the same type
    void update(const Display_properties_accumulator& source)
    {
        eccentricities.update(source.eccentricities.get_mean());
        spacings.update(source.spacings.get_mean());
        numbers_of_crowding_flankers.update(source.numbers_of_crowding_flankers.get_mean());
        spacings_of_crowding_flankers.update(source.spacings_of_crowding_flankers.get_mean());
        dbn_of_eccentricities.update(source.dbn_of_eccentricities);
        dbn_of_number_of_crowding_flankers.update(source.dbn_of_number_of_crowding_flankers);
        mean_n_crowders_for_eccentricity.update_with_bin_means(source.mean_n_crowders_for_eccentricity);
    }
    
    void output() const;
};


class Display_creator {
public:

    Display_creator();
    void initialize();
     // return a vector of Search_object, each fully constructed, positioned, and ready to be used
    Search_objects_t create_display(Condition_e condition, int set_size, Polarity_e polarity);
	void reset_display_statistics();	// call to reset accumulators (e.g. before each condition)
    void output_display_statistics() const;


private:
    // stimulus generation and representation
    using Locations_t = std::vector<GU::Point> ;
    Locations_t all_locations;
    

    Search_objects_t search_objects;
    
    Condition_e condition;
    Polarity_e polarity;
    int n_objects;
    
    Search_object target;
    
    
    // stimulus characteristic data calculated for each display
    // following is for characteristics measured from the fixation point -
    // corresponding to information available at the trial start, before any eye movements
    
    
    Display_properties_accumulator initial_display_properties;
    Display_properties_accumulator whole_display_properties;

    
    // helpers
    void generate_stimulus();
    void generate_CSF_stimulus();
    void generate_COC_stimulus();
    void generate_SHP_stimulus();
    void present_stimulus();
    void present_search_objects();
    void calculate_display_metrics();
    void calculate_display_statistics();
    void calculate_possible_next_fixation_statistics(GU::Point current_assumed_fixation_point, int possible_next_fixation_index, Display_properties_accumulator& properties_accumulator);
    void output_display_properties_accumulator(const char* message,const Display_properties_accumulator& properties_accumulator) const;
};


#endif /* DISPLAY_CREATOR_H */
